unit KpLib;

interface

uses
		{$IFNDEF WIN32}
		LongName,
		{$ENDIF}
		WinProcs, SysUtils, FileCtrl, Classes;

type
	BYTEPTR = ^Byte;

	TLFNFileStream = class(TFileStream)
	 public
		constructor Create( const FileName: string; Mode: Word);
	end;

function min( a,b: LongInt ): LongInt;
function max(a,b: LongInt): LongInt;
function BlockCompare(const Buf1, Buf2; Count: Integer): Boolean;

procedure ForceDirs(Dir: string);
function DirExists(Dir: string): Boolean;
function File_Exists(const FileName: string): Boolean;
procedure GetDirectory(D: Byte; var S: String);
procedure ChDirectory(const S: string);

{$IFDEF WIN32}
function StringAsPChar( var S: String): PChar;
{$ELSE}
procedure SetLength(var S: string; NewLength: Integer);
procedure ZeroMemory( p: Pointer; count: Integer );
procedure MoveMemory( dest,source: Pointer; count: Integer );
function GetEnvVar(EnvVar: String): String;
function GetTempPath( BufferSize: Integer; PathBuffer: PChar ): LongInt;
function StringAsPChar( var S: OpenString): PChar;
function ExtractFileDir(FName: String): String;
function LFN_CreateFile(FName: String ): LongBool;
function LFN_Shorten( LName: String ): String;
function LFN_GetShortFileName(LName: String): String;
function LFN_WIN31LongPathToShort(LName: String): String;
function LFN_LongPathToShort(LName: String): String;
function LFN_FileExists(LName: String): Boolean;
{$ENDIF}

var
	OSVersion: LongInt;

implementation

var
	DOSChars: array [0..77] of char;
const
	FNameChars: set of Char =
	['A'..'Z','a'..'z','0'..'9','_','^','$','~','!','#','%','&','-','{','}','@','`','''',')','('];

constructor TLFNFileStream.Create( const FileName: string; Mode: Word);
var
	FName: String;
begin
		FName := FileName;
	{$IFNDEF WIN32}
		If OSVersion > 3 then
		 begin
			If (Mode = fmCreate) then
				LFN_CreateFile( FName );
			FName := LFN_LongPathToShort( FName );
		 end
		Else
			FName := LFN_WIN31LongPathToShort(FName);
	{$ENDIF}
	inherited Create(FName,Mode);
end;

function min( a,b: LongInt ): LongInt;
begin
	If a < b then
  	Result := a
  Else
  	Result := b;
end;

function max(a,b: LongInt): LongInt;
begin
	If a > b then
  	Result := a
  Else
  	Result := b;
end;

procedure ForceDirs(Dir: string);
begin
	{$IFDEF WIN32}
	ForceDirectories(Dir);
	{$ELSE}
	If OSVersion > 3 then
	 begin
		if Dir[Length(Dir)] = '\' then
			SetLength(Dir, Length(Dir)-1);
		if (Length(Dir) < 3) or DirectoryExists(Dir) then Exit;
		ForceDirs(ExtractFilePath(Dir));
		W32CreateDirectory(StringAsPChar(Dir),nil,id_W32CreateDirectory);
	 end
	Else
	 begin
		Dir := LFN_WIN31LongPathToShort( Dir );
		ForceDirectories(Dir);
	 end;
	{$ENDIF}
end;

function File_Exists(const FileName: string): Boolean;
begin
	{$IFDEF WIN32}
	Result := FileExists(Filename);
	{$ELSE}
	 If OSVersion > 3 then
		Result := LFN_FileExists(Filename)
	 Else
		Result := FileExists(LFN_WIN31LongPathToShort(Filename));
	{$ENDIF}
end;

function DirExists(Dir: string): Boolean;
begin
	{$IFDEF WIN32}
	Result := DirectoryExists(Dir);
	{$ELSE}
	If OSVersion > 3 then
		Result := LFN_FileExists(Dir)
	Else
	 begin
		Dir := LFN_WIN31LongPathToShort( Dir );
		Result := DirectoryExists(Dir);
	 end;
	 {$ENDIF}
end;

procedure GetDirectory(D: Byte; var S: String);
var
	Drive: array[0..3] of Char;
	DirBuf, SaveBuf: array[0..259] of Char;
begin
	{$IFDEF WIN32}
	GetDir(D,S);
	{$ELSE}
	If OSVersion > 3 then
	 begin
		if D <> 0 then
		 begin
        Drive[0] := Chr(D + Ord('A') - 1);
        Drive[1] := ':';
        Drive[2] := #0;
			W32GetCurrentDirectory(SizeOf(SaveBuf), SaveBuf, id_W32GetCurrentDirectory);
			W32SetCurrentDirectory(Drive, id_W32SetCurrentDirectory);
		end;
	  W32GetCurrentDirectory(SizeOf(DirBuf), DirBuf, id_W32GetCurrentDirectory);
	  if D <> 0 then W32SetCurrentDirectory(SaveBuf, id_W32SetCurrentDirectory);
	  S := StrPas(@DirBuf);
	 end
	Else
		GetDir(D,S);  {We should never be Getting a long Dirname in Win31}
	{$ENDIF}
end;

procedure ChDirectory(const S: string);
var
	Dir: String;
begin
	{$IFDEF WIN32}
	ChDir(S);
	{$ELSE}
	If OSVersion > 3 then
	 begin
		Dir := S;
		W32SetCurrentDirectory(StringAsPChar(Dir),id_W32SetCurrentDirectory)
	 end
	Else
		ChDir(S);
	{$ENDIF}
end;

{$IFDEF WIN32}
function BlockCompare(const Buf1, Buf2; Count: Integer): Boolean;
type
	BufArray = array[0..MaxInt - 1] of Char;
var
  I: Integer;
begin
  Result := False;
  for I := 0 to Count - 1 do
    if BufArray(Buf1)[I] <> BufArray(Buf2)[I] then Exit;
  Result := True;
end;

function StringAsPChar( var S: String): PChar;
begin
  Result := PChar(S);
end;

{$ELSE}  { These functions are defined for 16 bit }
function BlockCompare(const Buf1, Buf2; Count: Integer): Boolean; assembler;
asm
        PUSH    DS
        LDS     SI,Buf1
        LES     DI,Buf2
        MOV     CX,Count
        XOR     AX,AX
        CLD
        REPE    CMPSB
        JNE     @@1
        INC     AX
@@1:    POP     DS
end;

procedure SetLength(var S: string; NewLength: Integer);
begin
	S[0] := Char(LoByte(NewLength));
end;

procedure ZeroMemory( p: Pointer; count: Integer );
var
	b: BYTEPTR;
	i: Integer;
begin
	b := BYTEPTR(p);
	for i := 0 to count-1 do
	 begin
		b^ := 0;
		Inc(b);
	 end;
end;

procedure MoveMemory( dest,source: Pointer; count: Integer );
var
	d,s: BYTEPTR;
  i: Integer;
begin
	d := BYTEPTR(dest);
	s := BYTEPTR(source);
	for i := 0 to count-1 do
	 begin
		d^ := s^;
		Inc(d);
		Inc(s);
	 end;
end;

function StringAsPChar( var S: OpenString): PChar;
begin
  If Length(S) = High(S) then
     Dec(S[0]);
  S[Ord(Length(S))+1] := #0;
  Result := @S[1];
end;

function GetEnvVar(EnvVar: String): String;
var
  P: PChar;
begin
  Result := '';
  P := GetDOSEnvironment;
  If Length(EnvVar) > 253 then
     SetLength(EnvVar, 253);
  EnvVar := EnvVar + '=';
	StringAsPChar(EnvVar);
  While P^ <> #0 do
		If StrLIComp(P, @EnvVar[1], Length(EnvVar)) <> 0 then
        Inc(P, StrLen(P)+1)
     Else
      begin
        Inc(P, Length(EnvVar));
        Result := StrPas(P);
        break;
      end;
end;

function GetTempPath( BufferSize: Integer; PathBuffer: PChar ): LongInt;
var
  thePath: String;
begin
  thePath := GetEnvVar( 'TMP' );
  If thePath = '' then
     thePath := GetEnvVar( 'TEMP' );
	If thePath = '' then
		GetDir(0,thePath);
	If thePath[Length(thePath)] <> '\' then
  	thePath := thePath + '\';
	StrPCopy( PathBuffer, thePath );
  Result := Length( thePath );
end;

function ExtractFileDir(FName: String): String;
begin
	Result := ExtractFilePath(FName);
end;

function LFN_CreateFile(FName: String): LongBool;
const
	GENERIC_READ             = $80000000;
	GENERIC_WRITE            = $40000000;
	CREATE_NEW 					 = 1;
	CREATE_ALWAYS 				 = 2;
	OPEN_EXISTING 				 = 3;
	OPEN_ALWAYS 				 = 4;
	TRUNCATE_EXISTING 		 = 5;
	FILE_ATTRIBUTE_NORMAL    = $00000080;
var
	theHandle: LongInt;
begin
	theHandle := W32CreateFile(StringAsPChar(FName),GENERIC_WRITE,0,nil,CREATE_ALWAYS,
										FILE_ATTRIBUTE_NORMAL,0,id_W32CreateFile);
	Result := W32CloseHandle( theHandle, id_W32CloseHandle );
end;

function LFN_GetShortFileName(LName: String): String;
var
	ffd: WIN32_FIND_DATA;
	r: LongInt;
begin
	r := W32FindFirstFile(StringAsPChar(LName),ffd,id_W32FindFirstFile);
	If (r  <> -1) and (StrPas(ffd.cAlternateFileName) <> '') then
		Result := ExtractFilePath(LName) + StrPas(ffd.cAlternateFileName)
	Else
		Result := LName;
	W32FindClose( r, id_W32FindClose );
end;

function hash( S: String; M: LongInt ): LongInt;
var
	i: Integer;
  g: LongInt;
begin
	Result := 0;
	for i := 1 to Length(S) do
	 begin
		Result := (Result shl 4) + Byte(S[i]);
		g := Result and $F0000000;
		If (g <> 0) then
			Result := Result xor (g shr 24);
		Result := Result and (not g);
	 end;
	 Result := Result mod M;
end;

function LFN_Shorten( LName: String ): String;
var
	i: Integer;
	Extent: String;
	HashChar: Char;
begin
	HashChar := #0;
	i := Length(LName);
	While (i > 0) do
	 begin
		If LName[i] = '.' then
			break;
		Dec(i);
	 end;
	If i > 0 then
	 begin
		If Length(LName)-i > 3 then
			HashChar := DOSChars[hash(LName,78)];
		Extent := Copy(LName,i,4);
		If HashChar <> #0 then
		 begin
			Extent[4] := HashChar;
			HashChar := #0;
		 end;
		If i > 9 then
			HashChar := DOSChars[hash(LName,78)];
		SetLength(LName, min(i-1,8));
		If HashChar <> #0 then
			LName[8] := HashChar;
	 end
	Else
	 begin
		Extent := '';
		If Length(LName) > 8 then
			HashChar := DOSChars[hash(LName,78)];
		SetLength(LName, min(Length(LName),8));
	 end;
	For i := 1 to Length(LName) do
		If not (LName[i] in FNameChars) then
			LName[i] := '_';
	Result := LName + Extent;
end;

function LFN_WIN31LongPathToShort(LName: String): String;
var
	tempShortPath: String;
  tmpStr: String;
	p: PChar;
	count, r, i, j: Integer;
	EndSlash: Boolean;
begin
	count := 0;
	EndSlash := False;
	tempShortPath := '';
	If (LName[2] = ':') and (LName[3] <> '\') then
		Insert('\',LName,3);
	If (LName[Length(LName)] = '\') then
	 begin
		EndSlash := True;
		Dec(LName[0]);
	 end;
	If (LName[1] = '\') then
		j := 2
	Else
		j := 1;

	For i := j to Length(LName) do
		If LName[i] = '\' then
		 begin
			LName[i] := #0;
			Inc(count);
		 end;
	LName[Length(LName)+1] := #0;
	p := @LName[j];
	If p[1] = ':' then
	 begin
		tempShortPath := StrPas(p) + '\';
		p := StrEnd(p);
		Inc(p);
		Dec(count);
	 end;
	For i := 0 to count do
	 begin
		tmpStr := StrPas(p);
		tmpStr := LFN_Shorten(tmpStr);
		tempShortPath := tempShortPath + tmpStr + '\';
		p := StrEnd(p);
		Inc(p);
	 end;
	If not EndSlash then
		Dec(tempShortPath[0]);
	Result := tempShortPath;
end;

function LFN_LongPathToShort(LName: String): String;
var
	tempLongPath: array [0..255] of char;
	tempShortPath: String;
	p: PChar;
	count, r, i, j: Integer;
	ffd: WIN32_FIND_DATA;
	EndSlash: Boolean;
begin
	count := 0;
  EndSlash := False;
	tempLongPath[0] := #0;
	If (LName[2] = ':') and (LName[3] <> '\') then
		Insert('\',LName,3);
	If (LName[Length(LName)] = '\') then
	 begin
		EndSlash := True;
		Dec(LName[0]);
	 end;
	If (LName[1] = '\') then
	 begin
		tempShortPath := '\';
		j := 2
	 end
	Else
		j := 1;
	For i := j to Length(LName) do
		If LName[i] = '\' then
		 begin
			LName[i] := #0;
			Inc(count);
		 end;
	LName[Length(LName)+1] := #0;
	p := @LName[j];
	If p[1] = ':' then
	 begin
		StrCopy(tempLongPath,p);
		StrCat(tempLongPath,'\');
		tempShortPath := StrPas(p) + '\';
		p := StrEnd(p);
		p^ := '\';
		Inc(p);
		Dec(count);
	 end;
	For i := 0 to count do
	 begin
		StrCat(tempLongPath,p);
		r := W32FindFirstFile(tempLongPath,ffd,id_W32FindFirstFile);
		if (r <> -1) and (StrPas(ffd.cAlternateFileName) <> '') then
			tempShortPath := tempShortPath +  StrPas(ffd.cAlternateFileName) + '\'
		Else
			tempShortPath := tempShortPath + StrPas(p) + '\';
		StrCat(tempLongPath,'\');
		p := StrEnd(p);
		p^ := '\';
		Inc(p);
		W32FindClose( r, id_W32FindClose);
	 end;
	If not EndSlash then
		Dec(tempShortPath[0]);
	Result := tempShortPath;
end;

function LFN_FileExists(LName: String): Boolean;
var
	ffd: WIN32_FIND_DATA;
	r: Integer;
begin
	If LName[Length(LName)] = '\' then
		Dec(LName[0]);
	r := W32FindFirstFile(StringAsPChar(LName),ffd,id_W32FindFirstFile);
	If r <> -1 then
		Result := True
	Else
		Result := False;
	W32FindClose( r, id_W32FindClose);
end;

{$ENDIF}

	{$IFNDEF WIN32}
var
	c: char;
	i: Integer;
begin
	OSversion := GetVersion;
	If (Lo(LOWORD(OSversion)) > 3) or
		((Lo(LOWORD(OSversion)) = 3) and (Hi(LOWORD(OSversion)) = 95)) then
		OSversion := 4   { WIN95 or higher }
	Else
		OSversion := 3;  { WIN31 }

	 {OSVersion := 3;}  { Uncomment this line to emulate WIN31 on WIN95 }
							  { Useful for testing WIN31 long filename support }
	   for c:= Low(Char) to High(Char) Do
			If c In FNameChars Then
			 begin
				DOSChars[i] := c;
				Inc(i);
			 end;
	{$ENDIF}
end.
